<?php

namespace App\Http\Controllers;

use App\Models\User;
use Illuminate\Http\Request;
use App\Http\Requests\AddUserRequest;
use App\Http\Requests\updateUserRequest;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Password;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Auth;


class AccountsController extends Controller
{
  /**
   * Display a listing of the resource.
   */
  public function index()
  {
    $stats = [
      ['label' => 'إجمالي الحسابات', 'value' => User::count(), 'bar' => 'bg-transparent', 'border' => 'progress-border-primary', 'progress' => 0],
      ['label' => 'صاحب مناسبة', 'value' => User::where('role', 'admin')->count(), 'bar' => 'bg-transparent', 'border' => 'progress-border-primary', 'progress' => 0],
      ['label' => 'حسابات المنظمين',   'value' => User::where('role', 'staff')->count(), 'bar' => 'bg-transparent', 'border' => 'progress-border-primary', 'progress' => 0],
      ['label' => 'حسابات المسؤول', 'value' => User::where('role', 'super-admin')->count(), 'bar' => 'bg-transparent', 'border' => 'progress-border-primary', 'progress' => 0],
    ];

    $eventOwners = User::where('role', '=', 'admin')->paginate(5);
    return view('theme.dashboard.accounts', compact('stats', 'eventOwners'));
  }

  /**
   * Show the form for creating a new resource.
   */
  public function create()
  {
    return view('theme.dashboard.addUser');
  }

  /**
   * Store a newly created resource in storage.
   */
  public function store(AddUserRequest $request)
  {
    DB::beginTransaction();

    $data = $request->validated();
    $data['phone_number'] = $request->country_code . $request->phone_number;
    $data['role'] = 'admin';
    $data['password'] = Str::random(32);

    $user = User::create($data);

    DB::commit();

    try {
      $token = Password::createToken($user);
      $user->sendPasswordResetNotification($token);
    } catch (\Throwable $mailException) {
      Log::warning('Password reset email failed', [
        'user_id' => $user->id,
        'error' => $mailException->getMessage(),
      ]);

      return redirect()
        ->route('accounts.index')
        ->with('userWarning', 'تم إضافة المستخدم، لكن فشل إرسال رابط تعيين كلمة المرور.');
    }

    return redirect()
      ->route('accounts.index')
      ->with('userSuccess', 'تم إضافة المستخدم وإرسال رابط تعيين كلمة المرور.');
  }


  /**
   * Display the specified resource.
   */
  public function show(User $account)
  {
    $events = $account->events()->latest()->paginate(5);
    return view('theme.dashboard.events', compact('events', 'account'));
  }

  /**
   * Show the form for editing the specified resource.
   */
  public function edit(User $account)
  {
    return view('theme.dashboard.editUser', compact('account'));
  }

  /**
   * Update the specified resource in storage.
   */

  public function update(updateUserRequest $request, User $account)
  {
    try {

      DB::beginTransaction();

      $account->update($request->validated());

      DB::commit();

      return redirect()
        ->route('accounts.index')
        ->with('userSuccess', 'تم تحديث المستخدم.');
    } catch (\Throwable $e) {

      DB::rollBack();

      Log::error('Update user failed', [
        'user_id' => $account->id,
        'error'   => $e->getMessage(),
      ]);

      return redirect()
        ->back()
        ->withInput()
        ->with('userError', 'حدث خطأ أثناء تحديث المستخدم، يرجى المحاولة لاحقًا.');
    }
  }


  /**
   * Remove the specified resource from storage.
   */

  public function destroy(User $account)
  {
    try {
      DB::beginTransaction();

      // ✅ منع حذف نفسك (اختياري)
      if (Auth::user()->id === $account->id) {
        return back()->with('userError', 'لا يمكنك حذف حسابك الحالي.');
      }

      // ✅ مهم: احذف الموارد عبر Laravel (عشان Model deleting يشتغل ويحذف الصور)
      $account->invitations()->each(fn($inv) => $inv->delete());
      $account->events()->each(fn($event) => $event->delete());

      // الآن احذف المستخدم
      $account->delete();

      DB::commit();

      return redirect()
        ->route('accounts.index')
        ->with('userSuccess', 'تم حذف المستخدم.');
    } catch (\Throwable $e) {

      DB::rollBack();

      Log::error('Delete user failed', [
        'user_id' => $account->id,
        'error'   => $e->getMessage(),
      ]);

      return back()->with('userError', 'حدث خطأ أثناء حذف المستخدم.');
    }
  }
}
