<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Http\Requests\AddEventRequest;
use Illuminate\Support\Facades\Auth;
use App\Models\Event;
use App\Http\Requests\GenerateInvitationsRequest;
use App\Models\Invitation;
use Illuminate\Support\Str;
use Endroid\QrCode\Builder\Builder;
use Endroid\QrCode\Writer\PngWriter;
use Illuminate\Support\Facades\Storage;
use Symfony\Component\HttpFoundation\File\UploadedFile;
use Endroid\QrCode\QrCode;
use App\Models\User;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;


class EventController extends Controller
{
  /**
   * Display a listing of the resource.
   */
  public function index()
  {
    $events = Event::where('user_id', Auth::user()->id)->orderBy("created_at", "desc")->paginate(10);
    return view('theme.events', compact('events'));
  }

  /**
   * Show the form for creating a new resource.
   */
  public function create()
  {
    return "create";
  }

  /**
   * Store a newly created resource in storage.
   */
  // public function store(AddEventRequest $request)
  // {

  //   $user = Auth::user();

  //   $createdEventsCount = $user->events->count();

  //   if ($createdEventsCount >= $user->events_limit) {
  //     return back()->with('addEventWarning', 'وصلت الحد المسموح لإنشاء المناسبات.');
  //   }

  //   // $data = $validatedRequest->validated();
  //   $data = $request->validated();

  //   // Image processing
  //   // 1- Get image 
  //   $image = $request->event_image;
  //   // 2- Generate image name
  //   $newImageName = time() . "_" . base64_encode($image->getClientOriginalName()) . "." . $image->getClientOriginalExtension();
  //   // dd($newImageName, $validatedRequest->file("image")->getClientOriginalName());
  //   // 3- Move image to public folder
  //   $image->storeAs("events/img", $newImageName, "public");
  //   // 4- Update image name in request
  //   // $validatedRequest->merge(["image" => $newImageName]);

  //   $data['event_image'] = $newImageName;
  //   $data['user_id'] = $user->id;
  //   $data['event_slug'] = Event::generateUniqueSlug($data['event_name']);

  //   // dd($validatedRequest);

  //   if (Event::create($data)) {
  //     return redirect()->back()->with("addEventSuccess", "تم إضافة المناسبة بنجاح");
  //   } else {
  //     return redirect()->back()->with("addEventError", "حدث خطأ أثناء إضافة المناسبة");
  //   }
  // }

  public function store(AddEventRequest $request)
  {
    $user = Auth::user();

    $createdEventsCount = $user->events->count();

    if ($createdEventsCount >= $user->events_limit) {
      return back()->with('addEventWarning', 'وصلت الحد المسموح لإنشاء المناسبات.');
    }

    $data = $request->validated();

    // سنحتفظ بمسار الصورة هنا لو تم رفعها
    $storedImagePath = null;

    try {

      DB::beginTransaction();

      // ✅ slug قبل الإنشاء
      $data['event_slug'] = Event::generateUniqueSlug($data['event_name']);
      $data['user_id']    = $user->id;

      // ✅ رفع الصورة (إن وُجدت) + تخزين المسار كامل في DB
      if ($request->hasFile('event_image')) {
        $image = $request->file('event_image');

        $newImageName = (string) Str::uuid() . '.' . $image->getClientOriginalExtension();
        $storedImagePath = $image->storeAs('events/img', $newImageName, 'public');
        // مثال: events/img/uuid.jpg

        $data['event_image'] = $storedImagePath; // ✅ خزّن المسار كامل
      }

      Event::create($data);

      DB::commit();

      return redirect()->back()->with('addEventSuccess', 'تم إضافة المناسبة بنجاح');
    } catch (\Throwable $e) {

      DB::rollBack();

      // ✅ لو الصورة اتخزنت وفشل إنشاء السجل، احذف الصورة
      if ($storedImagePath && Storage::disk('public')->exists($storedImagePath)) {
        Storage::disk('public')->delete($storedImagePath);
      }

      Log::error('Create event failed', [
        'user_id' => $user->id,
        'error'   => $e->getMessage(),
      ]);

      return redirect()->back()->with('addEventError', 'حدث خطأ أثناء إضافة المناسبة');
    }
  }

  /**
   * Display the specified resource.
   */
  public function show(string $id)
  {
    //
  }

  /**
   * Show the form for editing the specified resource.
   */
  public function edit(string $id)
  {
    //
  }

  /**
   * Update the specified resource in storage.
   */
  public function update(Request $request, string $id)
  {
    //
  }

  /**
   * Remove the specified resource from storage.
   */
  public function destroy(string $id)
  {
    //
  }

  public function generateInvitations(Event $event, GenerateInvitationsRequest $request)
  {
    $validated = $request->validated();
    $count = (int) $validated['invitation_count'];

    if ($count > Auth::user()->user_invitation_limit) {
      return redirect()->back()->with("generateInvitationsError", "عدد الدعوات المطلوبة أكبر من الحد المسموح");
    }

    if ($count <= 0) {
      return redirect()->back()->with("generateInvitationsError", "عدد الدعوات المطلوبة يجب أن يكون أكبر من 0");
    }

    if (Auth::user()->user_invitation_limit - $event->invitations->count() < $count) {
      return redirect()->back()->with("generateInvitationsError", "عدد الدعوات المتبقية غير كافية وصلت الحد الأقصى للدعوات تواصل معنا في حالة طلب المزيد");
    }

    $event_id = $event->id;

    // (اختياري) حفظ جماعي أسرع
    $rows = [];

    $slugFolder = 'qr_codes/' . $event->event_slug;

    // إذا المجلد غير موجود → أنشئه
    if (!Storage::disk('public')->exists($slugFolder)) {
      Storage::disk('public')->makeDirectory($slugFolder);
    }

    for ($i = 0; $i < $count; $i++) {

      $qr_token = (string) Str::uuid();

      // الأفضل تخليه رابط كامل أو path واضح (حسب نظامك)
      $invitation_url = $event->event_slug;
      // dd($invitation_url);

      $qr_image_path = $slugFolder . '/' . $qr_token . '.png';

      // إنشاء QR
      $qrCode = new QrCode(
        data: config('app.url') . "/invitation/" . $qr_token,
        size: 600,
        margin: 10
      );

      // Writer
      $writer = new PngWriter();
      $result = $writer->write($qrCode);

      // حفظ الصورة
      Storage::disk('public')->put($qr_image_path, $result->getString());

      $rows[] = [
        'event_id' => $event_id,
        'qr_token' => $qr_token,
        'invitation_url' => $invitation_url . '_' . $qr_token,
        'qr_image_path' => $qr_image_path,
        'user_id' => $event->user_id,
        'created_at' => now(),
        'updated_at' => now(),
      ];
    }

    Invitation::insert($rows);
    return redirect()
      ->route('theme.invitations', $event->event_slug)
      ->with(
        'generateInvitationsSuccess',
        $count == 1 ? "تم إنشاء دعوة بنجاح" : "تم إنشاء {$count} دعوات بنجاح"
      );
  }
}
