<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Storage;

class Event extends Model
{
  const STATUS_DRAFT = 'draft';
  const STATUS_PUBLISHED = 'published';
  const STATUS_CLOSED = 'closed';

  protected $fillable = [
    'event_name',
    'event_date',
    'event_time',
    'event_location',
    'event_image',
    'event_slug',
    'user_id',
    'location_url',
    'event_status',
  ];

  protected $casts = [
    'user_id' => 'integer',
  ];




  protected static function booted()
  {
    static::deleting(function ($event) {

      // حذف صورة المناسبة (الآن محفوظة كمسار كامل: events/img/xxx.jpg)
      if (!empty($event->event_image) && Storage::disk('public')->exists($event->event_image)) {
        Storage::disk('public')->delete($event->event_image);
      }

      // حذف مجلد QR الخاص بالمناسبة (إذا كنت تحفظ QR داخل qr_codes/{slug}/...)
      if (!empty($event->event_slug)) {
        Storage::disk('public')->deleteDirectory('qr_codes/' . $event->event_slug);
      }

      // لو عندك invitations مرتبطة بالمناسبة، احذفها عبر Laravel حتى تشتغل deleting في Invitation
      if (method_exists($event, 'invitations')) {
        $event->invitations()->each(fn($inv) => $inv->delete());
      }
    });
  }
  // relations
  // user relation
  public function user()
  {
    return $this->belongsTo(User::class, 'user_id');
  }
  // invitations relation
  public function invitations()
  {
    return $this->hasMany(Invitation::class);
  }



  // generate unique slug for event
  public static function generateUniqueSlug(string $title): string
  {
    $slug = Str::slug($title);
    $originalSlug = $slug;
    $count = 1;

    while (self::where('event_slug', $slug)->exists()) {
      $slug = $originalSlug . '-' . $count;
      $count++;
    }

    return $slug;
  }
}
